#include "otpch.h"
#include <set>
#include "itemattributes.h"

// Configuração dos atributos esperados no Lua
static struct I {
    const char* name;
    int type;
} itemrarityattributes[] = {
    { "id", LUA_TNUMBER },
    { "type", LUA_TTABLE },
    { "min", LUA_TUSERDATA },
    { "max", LUA_TUSERDATA },
    { "chance", LUA_TNUMBER },
    { nullptr, 0 }
};

// Função para carregar as chances de raridade
bool ItemRarityAttributes::loadChances(lua_State* L)
{
    auto getChances = [&](const char* name, bool fromMonster) {
        lua_getfield(L, -1, name);
        if (!lua_istable(L, -1)) {
            return false;
        }

        lua_pushnil(L);
        while (lua_next(L, -2)) {
            lua_getfield(L, -1, "id");
            if (!lua_isnumber(L, -1)) {
                lua_pop(L, 1);
                return false;
            }

            ItemRarity_t rarityId = static_cast<ItemRarity_t>(lua_tointeger(L, -1));
            lua_pop(L, 1);

            lua_getfield(L, -1, "chance");
            if (!lua_isnumber(L, -1)) {
                lua_pop(L, 1);
                return false;
            }

            int32_t chance = static_cast<int32_t>(lua_tointeger(L, -1));
            lua_pop(L, 1);

            m_chances[fromMonster][rarityId] = chance;
            lua_pop(L, 1);
        }

        lua_pop(L, 1);
        return true;
    };

    lua_getglobal(L, "RARITY_CHANCE");

    if (!getChances("fromMonster", true) || !getChances("fromQuest", false)) {
        lua_pop(L, 1);
        return false;
    }

    lua_pop(L, 1);
    return true;
}

// Função para carregar os modificadores de raridade
bool ItemRarityAttributes::loadModifiers(lua_State* L)
{
    lua_getglobal(L, "RARITY_MODIFIERS");
    for (uint16_t i = 1;; i++) {
        lua_rawgeti(L, -1, i);
        if (lua_isnil(L, -1)) {
            lua_pop(L, 1);
            break;
        }

        lua_getfield(L, -1, "id");
        if (!lua_isnumber(L, -1)) {
            lua_pop(L, 1);
            return false;
        }

        ItemRarity_t rarityId = static_cast<ItemRarity_t>(lua_tointeger(L, -1));
        lua_pop(L, 1);

        lua_getfield(L, -1, "amount");
        if (!lua_isnumber(L, -1)) {
            lua_pop(L, 1);
            return false;
        }

        int32_t amount = lua_tointeger(L, -1);
        lua_pop(L, 1);

        m_modifiers[rarityId] = amount;
        lua_pop(L, 1);
    }

    return true;
}

bool ItemRarityAttributes::loadAttributes(lua_State* L)
{
    try {
        lua_getglobal(L, "RARITY_ATTRIBUTES_ITEMS");
        if (!lua_istable(L, -1)) {
            std::cout << "RARITY_ATTRIBUTES_ITEMS não encontrado ou não é uma tabela." << std::endl;
            lua_pop(L, 1);
            return false;
        }

        lua_pushnil(L);
        while (lua_next(L, -2)) {
            if (!lua_istable(L, -1)) {
                lua_pop(L, 1);
                continue;
            }

            std::vector<int32_t> itemIds;
            lua_getfield(L, -1, "items");
            if (lua_istable(L, -1)) {
                lua_pushnil(L);
                while (lua_next(L, -2)) {
                    if (lua_isnumber(L, -1)) {
                        itemIds.push_back(static_cast<int32_t>(lua_tointeger(L, -1)));
                    }
                    lua_pop(L, 1);
                }
            }
            lua_pop(L, 1);

            std::vector<ItemAttributeInfo> attributes;
            lua_getfield(L, -1, "attributes");
            if (lua_istable(L, -1)) {
                lua_pushnil(L);
                while (lua_next(L, -2)) {
                    if (!lua_istable(L, -1)) {
                        lua_pop(L, 1);
                        continue;
                    }

                    ItemAttributeInfo attributeInfo;
                    lua_getfield(L, -1, "id");
                    if (!lua_isnumber(L, -1)) {
                        lua_pop(L, 1);
                        continue;
                    }
                    attributeInfo.id = static_cast<ItemTooltipAttributes_t>(lua_tointeger(L, -1));
                    lua_pop(L, 1);

                    lua_getfield(L, -1, "min");
                    attributeInfo.min = lua_isnumber(L, -1) ? static_cast<int32_t>(lua_tonumber(L, -1)) : 0;
                    lua_pop(L, 1);

                    lua_getfield(L, -1, "max");
                    attributeInfo.max = lua_isnumber(L, -1) ? static_cast<int32_t>(lua_tonumber(L, -1)) : 0;
                    lua_pop(L, 1);

                    lua_getfield(L, -1, "type");
                    if (lua_istable(L, -1)) {
                        lua_pushnil(L);
                        while (lua_next(L, -2)) {
                            if (lua_isnumber(L, -1)) {
                                int32_t type = static_cast<int32_t>(lua_tointeger(L, -1));
                                attributeInfo.types.push_back(type);
                                for (int32_t itemId : itemIds) {
                                    m_skillItems[type].push_back(itemId);
                                }
                            }
                            lua_pop(L, 1);
                        }
                    }
                    lua_pop(L, 1);

                    attributes.push_back(attributeInfo);
                    lua_pop(L, 1);
                }
            }
            lua_pop(L, 1);

            for (int32_t itemId : itemIds) {
                m_allowedAttributes[itemId] = attributes;
            }

            lua_pop(L, 1);
        }
        lua_pop(L, 1);
        return true;
    } catch (const std::exception& e) {
        std::cout << "Exception in loadAttributes: " << e.what() << std::endl;
    } catch (...) {
        std::cout << "Unknown error occurred in loadAttributes." << std::endl;
    }
    return false;
}


bool ItemRarityAttributes::load()
{
    lua_State* L = luaL_newstate();
    if (!L) {
        throw std::runtime_error("Failed to allocate memory in ItemAttributes");
    }

    luaL_openlibs(L);
    LuaScriptInterface::registerEnums(L);

    if (luaL_dofile(L, "data/LUA/rarityAttributes.lua")) {
        lua_close(L);
        return false;
    }

    if (!loadChances(L) || !loadModifiers(L) || !loadAttributes(L)) {
        lua_close(L);
        return false;
    }

    lua_close(L);
    return true;
}

double ItemRarityAttributes::getRarityChance() const
{
    lua_State* L = luaL_newstate();
    if (!L) {
        throw std::runtime_error("Failed to allocate memory in ItemRarityAttributes::getRarityChance");
    }

    luaL_openlibs(L);
    LuaScriptInterface::registerEnums(L);

    if (luaL_dofile(L, "data/LUA/rarityAttributes.lua")) {
        lua_close(L);
        return 0.0;
    }

    lua_getglobal(L, "RARITY_CHANCE");
    lua_getfield(L, -1, "ondrop");

    double ondropChance = 0.0;
    if (lua_isnumber(L, -1)) {
        ondropChance = lua_tonumber(L, -1); // Agora aceita valores decimais
    }

    lua_pop(L, 2);
    lua_close(L);

    return ondropChance;
}


ItemRarity_t ItemRarityAttributes::getRandomRarityId(bool fromMonster) const
{
    auto itChances = m_chances.find(fromMonster);
    if (itChances == m_chances.end()) {
        return ITEM_RARITY_NONE;
    }

    int32_t totalChance = 0;
    for (const auto& itChance : itChances->second) {
        totalChance += itChance.second;
    }

    int32_t randomValue = uniform_random(0, totalChance);

    int32_t cumulativeChance = 0;
    for (const auto& itChance : itChances->second) {
        cumulativeChance += itChance.second;
        if (randomValue <= cumulativeChance) {
            return itChance.first;
        }
    }

    return ITEM_RARITY_COMMON;
}

bool ItemRarityAttributes::setRandomAttributes(ItemRarity_t rarityId,
    std::multimap<ItemTooltipAttributes_t, std::pair<int32_t, IntegerVector>>* itemAttributes,
    int32_t itemId)
{
    try {
        // Adicionado: Verifica se o itemId está na tabela antes de tudo
        auto itAllowedAttributes = m_allowedAttributes.find(itemId);
        if (itAllowedAttributes == m_allowedAttributes.end()) {
            return false; // Se não está, nem tenta adicionar raridade
        }

        double ondropChance = getRarityChance();
        double randomRoll = uniform_random(0.0, 100.0);
        if (randomRoll > ondropChance) {
            return false;
        }

        if (rarityId == ITEM_RARITY_NONE) {
            rarityId = getRandomRarityId(true);
        }
        if (rarityId == ITEM_RARITY_NONE) {
            return false;
        }

        itemAttributes->clear();

        const auto& allowedAttributes = itAllowedAttributes->second;
        auto itModifiers = m_modifiers.find(rarityId);
        if (itModifiers == m_modifiers.end() || allowedAttributes.empty()) {
            return false;
        }
        int maxAttributes = itModifiers->second;

        struct Entry {
            const ItemAttributeInfo* info;
            int type;
        };
        std::vector<Entry> entries;

        for (const auto& attr : allowedAttributes) {
            if (!attr.types.empty()) {
                for (int t : attr.types) {
                    auto itSkill = m_skillItems.find(t);
                    if (itSkill != m_skillItems.end() &&
                        std::find(itSkill->second.begin(), itSkill->second.end(), itemId) != itSkill->second.end()) {
                        entries.push_back({ &attr, t });
                    }
                }
            } else {
                entries.push_back({ &attr, -1 });
            }
        }

        if (entries.empty()) {
            return false;
        }

        std::vector<Entry> selectedEntries;
        if (entries.size() <= static_cast<size_t>(maxAttributes)) {
            selectedEntries = entries;
        } else {
            std::shuffle(entries.begin(), entries.end(), std::default_random_engine(std::random_device{}()));
            selectedEntries.assign(entries.begin(), entries.begin() + maxAttributes);
        }

        for (const auto& e : selectedEntries) {
            std::random_device rd;
            std::mt19937 gen(rd());
            std::uniform_int_distribution<int32_t> distrib(e.info->min, e.info->max);
            int32_t value = distrib(gen);

            if (e.info->id == TOOLTIP_ATTRIBUTE_STATS) {
                if (e.type == STAT_MAXHITPOINTS || e.type == STAT_MAXMANAPOINTS) {
                    value = (value / 5) * 5;
                    if (value < e.info->min) {
                        value += 5;
                    }
                    if (value > e.info->max) {
                        value -= 5;
                    }
                }
            } else if (e.info->id == TOOLTIP_ATTRIBUTE_SPEED) {
                value = (value / 5) * 5;
                if (value < e.info->min) {
                    value += 5;
                }
                if (value > e.info->max) {
                    value -= 5;
                }
            }

            IntegerVector types;
            if (e.type != -1) {
                types.push_back(e.type);
            }

            itemAttributes->emplace(e.info->id, std::make_pair(value, types));
        }

        return !itemAttributes->empty();
    } catch (const std::exception& e) {
        std::cout << "Exception in setRandomAttributes: " << e.what() << std::endl;
    } catch (...) {
        std::cout << "Unknown error occurred in setRandomAttributes." << std::endl;
    }
    return false;
}
